#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include "resource.h"
#include "song.h"

#define TRACK_DONT_TOUCH		0
#define TRACK_AUTO_INCREMENT	1
#define TRACK_FROM_TITLE		2

#define TRACK_HEADER			0
#define TRACK_FOOTER			1
#define TRACK_NEW				2

#define get_selected() SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCURSEL, 0, 0)
#define get_selected_data() SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, get_selected(), 0)
#define get_count() SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCOUNT, 0, 0)

#define DEF_FOOTER "<hr /><i>Created by <a href=\"http://gdr.prv.pl/\" target=\"_blank\">GDR!</a>'s LyricGen</i><br />\r\n"
#define DEF_TITLE "Lyrics file created by GDR!'s LyricGen"

int currtrack;
char *header = NULL, *footer = NULL, *title = NULL;


/* Shows a debug message */
void MBox(char *msg)
{
	MessageBox(NULL, msg, "Debug message", 0);
}


/* A secure malloc() */
void *smalloc(int size)
{
	void *p = calloc(size, 1);
	if(!p)
	{
		MessageBox(0, "Ran out of memory!", "Warning", 0);
		/* (shutdown) */
	}
	return p;

}


/* Initialize the main dialog */
void init_dialog(HWND hwnd)
{
	int index;
	index = SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_ADDSTRING, 0, (LPARAM)"Auto increment track #");
	SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_SETCURSEL, 0, (LPARAM)index);
	SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_SETITEMDATA, index, (LPARAM)TRACK_AUTO_INCREMENT);

	index = SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_ADDSTRING, 0, (LPARAM)"Take track # from title");
	SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_SETITEMDATA, index, (LPARAM)TRACK_FROM_TITLE);

	index = SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_ADDSTRING, 0, (LPARAM)"Manually enter track #");
	SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_SETITEMDATA, index, (LPARAM)TRACK_DONT_TOUCH);

	index = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_ADDSTRING, 0, (LPARAM)"(Header)");
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, index, (LPARAM)TRACK_HEADER);
	
	index = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_ADDSTRING, 0, (LPARAM)"(Footer)");
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, index, (LPARAM)TRACK_FOOTER);
	
	index = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_ADDSTRING, 0, (LPARAM)"(New track)");
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, index, (LPARAM)TRACK_NEW);

	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETCURSEL, index, 0);

	SetDlgItemText(hwnd, IDC_TRACK, "1");

	footer = smalloc(strlen(DEF_FOOTER)+1);
	strcpy(footer, DEF_FOOTER);

	title = smalloc(strlen(DEF_TITLE)+1);
	strcpy(title, DEF_TITLE);
}


/* Gets title from first line of lyrics */
void get_title(song *s)
{
	char *p;

	if(s->title) free(s->title);
	s->titlelen = 0;
	p = s->lyrics;
	while(*p != '\n') p++;
	s->title = smalloc(p - s->lyrics);
	strncpy(s->title, s->lyrics, p - s->lyrics - 1);
	s->titlelen = strlen(s->title);
}


/* Gets track number from title */
void get_track(song *s)
{
	char *p = s->title;
	do
	{
		int i = atoi(p);
		if(i)
		{
			s->tracknum = i;
			break;
		}
	} while(*(++p));
}


/* Checks whether "Get track # from title" is checked */
int is_track_from_title(HWND hwnd)
{
	int i = SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_GETCURSEL, 0, 0);
	switch(SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_GETITEMDATA, i, 0))
	{
	case TRACK_FROM_TITLE:
		return 1;
	default:
		return 0;
	}
}


/* Checks whether "Auto increment track #" is checked */
int is_track_autoincrement(HWND hwnd)
{
	int i = SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_GETCURSEL, 0, 0);
	switch(SendDlgItemMessage(hwnd, IDC_NUMBERTYPE, CB_GETITEMDATA, i, 0))
	{
	case TRACK_AUTO_INCREMENT:
		return 1;
	default:
		return 0;
	}
}


/* Frees memory of song structure */
void free_song(song *s)
{
	if(!s) return;
	if(s->lyrics) free(s->lyrics);
	if(s->title) free(s->title);
	free(s);
}


int my_isspace(int c)
{
	if(isspace(c)) return 1;
	if(strchr("-_=", c)) return 1;
	return 0;
}


/* removes white chars from beginning of lyrics */
void correct_lyrics(char *l)
{
	char *p = l;
	while(*(p++)!='\n');
	while(my_isspace(*(++p)));
	memmove(l, p, strlen(p)+1);
}

int my_isspace2(int c)
{
	if(isspace(c)) return 1;
	if(strchr("-_.", c)) return 1;
	return 0;
}

int is_numeric(int c)
{
	if(strchr("0123456789#-", c)) return 1;
	return 0;
}


/* Removes track # if detected at the beginning of title */
void correct_title(char *l)
{
	char *p = l;
	while(is_numeric(*(++p)));
	while(my_isspace2(*(++p)));
	if(p != l) memmove(l, p, strlen(p)+1);
	else
	{
		//try to remove number from end of title...
	}
}

/* Counts occurences of c in string s */
int count_chars(char *s, char c)
{
	int cnt = 0;

	while(*s)
	{
		if(*s == c) cnt++;
		s++;
	}
	return cnt;
}


/* Converts newline to <br /> tag */
void nl2br(char **s)
{
	char *p, *q, *d;
	int i;

	i = count_chars(*s, '\n');
	d = smalloc(strlen(*s) + 9*i + 1);

	p = *s;
	q = d;
	while(*p)
	{
		if((*p == '\r') || (*p == '\n'))
		{
			strcpy(q, "<br />\n");
			q += strlen("<br />\n");
			if(*p == '\r') p++;	// \n
		}
		else
		{
			*q = *p;
			q++;
		}
		p++;
	}
	*q = 0;

	free(*s);
	*s = d;
}


/* Called when Add is selected on "New track" */
void add_track_new(HWND hwnd, song *s)
{
	char *t;
	int i;

	if(IsDlgButtonChecked(hwnd, IDC_GETTITLE))
	{
		get_title(s);
		correct_lyrics(s->lyrics);
	}

	if(is_track_from_title(hwnd))
	{
		get_track(s);
		correct_title(s->title);
	}
	else if(is_track_autoincrement(hwnd))
	{
		t = smalloc(15);
		sprintf(t, "%d", s->tracknum + 1);
		SetDlgItemText(hwnd, IDC_TRACK, t);
		free(t);
	}

	t = smalloc(strlen(s->title) + 10);
	sprintf(t, "%3d. %s", s->tracknum, s->title);
	i = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCOUNT, 0, 0);
	i = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_INSERTSTRING, i-2, (LPARAM)t);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, i, (LPARAM)s);
	free(t);
}


/* Called when Add is clicked on "Header" */
void add_header(HWND hwnd, song *s)
{
	int i;
	HWND h;

	if(header) free(header);

	h = GetDlgItem(hwnd, IDC_LYRICS);
	i = GetWindowTextLength(h);
	header = smalloc(i+1);
	GetWindowText(h, header, i+1);

	if(title) free(title);
	h = GetDlgItem(hwnd, IDC_TITLE);
	i = GetWindowTextLength(h);
	title = smalloc(i+1);
	GetWindowText(h, title, i+1);
}

/* Called when Add is clicked on "Footer" */
void add_footer(HWND hwnd, song *s)
{
	int i;
	HWND h;
	if(footer) free(footer);

	h = GetDlgItem(hwnd, IDC_LYRICS);
	i = GetWindowTextLength(h);
	footer = smalloc(i+1);
	GetWindowText(h, footer, i+1);
}


/* Connects struct song with tracklist entry number i */
void set_track_data(HWND hwnd, song *s, int i)
{
	char *t;
	long data;

    data = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i, 0);
	if(data != LB_ERR)
		free_song((song *)data);
	t = smalloc(s->titlelen + 10);
	sprintf(t, "%3d. %s", s->tracknum, s->title);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_DELETESTRING, i, 0);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_INSERTSTRING, i, (LPARAM)t);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, i, (LPARAM)s);
	free(t);
}


/* Edit is clicked on a regular track */
void add_track_edit(HWND hwnd, song *s)
{
	int i = get_selected();

	set_track_data(hwnd, s, i);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETCURSEL, i, 0);
}


/* Creates instance of struct song filled with data from dialog */
song *save_track(HWND hwnd)
{
	song *s = malloc(sizeof(song));
	int i;
	HWND h;
	char *t;

	h = GetDlgItem(hwnd, IDC_LYRICS);
	s->lyricslen = GetWindowTextLength(h);
	s->lyrics = smalloc(s->lyricslen+1);
	GetWindowText(h, s->lyrics, s->lyricslen+1);

	h = GetDlgItem(hwnd, IDC_TITLE);
	s->titlelen = GetWindowTextLength(h);
	s->title = smalloc(s->titlelen+1);
	GetWindowText(h, s->title, s->titlelen+1);

	h = GetDlgItem(hwnd, IDC_TRACK);
	i = GetWindowTextLength(h);
	t = smalloc(i+1);
	GetWindowText(h, t, i+1);
	s->tracknum = atoi(t);
	free(t);

	return s;
}


/* Called when "Add" / "Edit" button is clicked */
void add_song(HWND hwnd)
{
	song *s;
	s = save_track(hwnd);

	if(get_selected_data() == TRACK_NEW)
	{
		add_track_new(hwnd, s);
	}
	else if(get_selected_data() == TRACK_HEADER)
	{
		add_header(hwnd, s);
	}
	else if(get_selected_data() == TRACK_FOOTER)
	{
		add_footer(hwnd, s);
	}
	else
	{
		add_track_edit(hwnd, s);
	}
}


/* Asks user to confirm */
int confirm(HWND hwnd, char *query)
{
	int r = MessageBox(hwnd, query, "Question", MB_YESNO | MB_ICONEXCLAMATION);
	return (r==IDYES)?1:0;
}


/* Compares string c with string from a given dialog item */
int strdlgcmp(HWND hwnd, char *c, int handle)
{
	HWND h;
	char *t;
	int i;

	h = GetDlgItem(hwnd, handle);
	i = GetWindowTextLength(h);
	t = smalloc(i+1);
	GetWindowText(h, t, i+1);

	i = strcmp(c, t);
	free(t);
	return i;
}

int content_changed(HWND hwnd)
{
	song *s = (song *)SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, currtrack, 0);
	if(strdlgcmp(hwnd, s->lyrics, IDC_LYRICS)) return 1;
	if(strdlgcmp(hwnd, s->title, IDC_TITLE)) return 1;
//	if(strdlgcmp(hwnd, s->tracknum, IDC_TRACK)) return 1;

	return 0;
}


/* Enables / disables dialog item */
void item_state(HWND hwnd, int id, int state)
{
		HWND i;
		i = GetDlgItem(hwnd, id);
		EnableWindow(i, state);
}


/* strcpy() with allocation */
void sstrcpy(char **d, char *s)
{
	if(*d) free(*d);

	*d = smalloc(strlen(s)+1);
	strcpy(*d, s);
}


/* Called when selection changes on tracklist */
void sel_change(HWND hwnd)
{
	int i;
	long data;
	char *p;
	song *s;

	long trks = get_count();
	/* save current track... */
	s = save_track(hwnd);
	if((currtrack > 0) && (currtrack < trks - 2)) 
	{
		set_track_data(hwnd, s, currtrack);
	}
	else 
	{
		if(currtrack == trks - 2)
		{
			/* footer */
			sstrcpy(&footer, s->lyrics);
		}
		else if(currtrack == 0)
		{
			/* header */

			sstrcpy(&header, s->lyrics);
			sstrcpy(&title, s->title);
		}
		free(s);
	}

	i = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCURSEL, 0, 0);
	if(i == currtrack) return;
	data = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i, 0);
	if(data == TRACK_HEADER)
	{
		if(header) SetDlgItemText(hwnd, IDC_LYRICS, header);
		else SetDlgItemText(hwnd, IDC_LYRICS, "");
		if(title) SetDlgItemText(hwnd, IDC_TITLE, title);
		else SetDlgItemText(hwnd, IDC_TITLE, "");
		SetDlgItemText(hwnd, IDOK, "Edit");
		item_state(hwnd, IDC_TITLE, TRUE);
		item_state(hwnd, IDC_TRACK, FALSE);
	}
	else if(data == TRACK_FOOTER)
	{
		if(footer) SetDlgItemText(hwnd, IDC_LYRICS, footer);
		else SetDlgItemText(hwnd, IDC_LYRICS, "");
		SetDlgItemText(hwnd, IDOK, "Edit");
		item_state(hwnd, IDC_TITLE, FALSE);
		item_state(hwnd, IDC_TRACK, FALSE);
	}
	else if(data == TRACK_NEW)
	{
		SetDlgItemText(hwnd, IDC_TITLE, "");
		SetDlgItemText(hwnd, IDC_LYRICS, "");

		SetDlgItemText(hwnd, IDOK, "Add");
		item_state(hwnd, IDC_TITLE, TRUE);
		item_state(hwnd, IDC_TRACK, TRUE);
	}
	else
	{

		/* ...and restore the clicked one */
		s = (song *)data;
		SetDlgItemText(hwnd, IDC_TITLE, s->title);
		SetDlgItemText(hwnd, IDC_LYRICS, s->lyrics);
		p = malloc(15);
		sprintf(p, "%d", s->tracknum);
		SetDlgItemText(hwnd, IDC_TRACK, p);
		free(p);

		SetDlgItemText(hwnd, IDOK, "Edit");
		item_state(hwnd, IDC_TITLE, TRUE);
		item_state(hwnd, IDC_TRACK, TRUE);
	}
	currtrack = get_selected();
}


/* Called when "Remove" button is clicked */
void remove_track(HWND hwnd)
{
	int i;
	long data;
	song *s;

	i = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCURSEL, 0, 0);
	data = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i, 0);
	if(data == TRACK_HEADER)
	{
		if(confirm(hwnd, "Do you really want to erase header contents?"))
		{
			sprintf(header, "");
			SetDlgItemText(hwnd, IDC_LYRICS, "");
			sprintf(title, "");
			SetDlgItemText(hwnd, IDC_TITLE, "");
		}
	}
	else if(data == TRACK_FOOTER)
	{
		if(confirm(hwnd, "Do you really want to erase footer contents?"))
		{
			sprintf(footer, "");
			SetDlgItemText(hwnd, IDC_LYRICS, "");
		}
	}
	else if(data == TRACK_NEW)
	{
		SetDlgItemText(hwnd, IDC_TITLE, "");
		SetDlgItemText(hwnd, IDC_LYRICS, "");
	}
	else
	{
		if(!confirm(hwnd, "Do you really want to erase lyrics for this song?"))
			return;
		s = (song *)data;
		free_song(s);
		SetDlgItemText(hwnd, IDC_TITLE, "");
		SetDlgItemText(hwnd, IDC_LYRICS, "");
		SendDlgItemMessage(hwnd, IDC_TRACKS, LB_DELETESTRING, i, 0);
	}
}


/* Swaps two entries in a tracks listbox */
void list_swap(HWND hwnd, int i1, int i2)
{
	int len = SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETTEXTLEN, i1, 0);
	char *b = malloc(len+1);
	song *s;

	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETTEXT, i1, (LPARAM)b);
	s = (song *) SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i1, 0);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_DELETESTRING, i1, 0);

	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_INSERTSTRING, i2, (LPARAM)b);
	SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETITEMDATA, i2, (LPARAM)s);
}


/* Moves selected song up in the tracklist */
void move_up(HWND hwnd)
{
	int sel = get_selected();
	if((sel > 1) && (sel < get_count() - 2))
	{
		list_swap(hwnd, sel, sel-1);
		SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETCURSEL, sel-1, 0);
	}
}


/* Moves selected song down in the tracklist */
void move_down(HWND hwnd)
{
	int sel = get_selected();
	if((sel > 0) && (sel < get_count() - 3))
	{
		list_swap(hwnd, sel, sel+1);
		SendDlgItemMessage(hwnd, IDC_TRACKS, LB_SETCURSEL, sel+1, 0);
	}
}


void put_headers(HWND hwnd, FILE *f)
{
	fprintf(f, "<html>\n<head>\n<title>%s</title>\n<meta name=\"generator\" value=\"GDR!'s LyricGen\">\n</head>\n<body>\n", title);
}

void put_header(HWND hwnd, FILE *f)
{
	if(header) fprintf(f, "<div class=\"header\">\n%s\n</div>\n", header);
}

void put_footer(HWND hwnd, FILE *f)
{
	if(footer) fprintf(f, "<div class=\"footer\">\n%s\n</div>\n", footer);
}

void put_footers(HWND hwnd, FILE *f)
{
	fprintf(f, "</body>\n</html>");
}

void put_song(HWND hwnd, FILE *f, song *s)
{
	/* create a copy of string and nl2br it */
	char *p = smalloc(strlen(s->lyrics)+1);
	strcpy(p, s->lyrics);
	nl2br(&p);
	fprintf(f, "<div class=\"song\">\n<a name=\"s%d\"></a>\n <h2>%d. %s</h2>\n%s\n</div>\n\n", s->tracknum, s->tracknum, s->title, p);
}

void put_tracklist_header(HWND hwnd, FILE *f)
{
	fprintf(f, "<div class=\"toc\">\n<ul>\n");
}

void put_tracklist_entry(HWND hwnd, FILE *f, song *s)
{
	fprintf(f, "<li><a href=\"#s%d\">%d. %s</a>\n", s->tracknum, s->tracknum, s->title);
}

void put_tracklist_footer(HWND hwnd, FILE *f)
{
	fprintf(f, "</ul>\n</div>");
}


/* Called if "Generate" button is clicked */
void generate_output(HWND hwnd)
{
	int i;
	FILE *f;

	OPENFILENAME ofn;
	char szFileName[MAX_PATH] = "";

	ZeroMemory(&ofn, sizeof(ofn));

	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hwnd;
	ofn.lpstrFilter = "HTML Files(*.html)\0*.html\0All Files (*.*)\0*.*\0";
	ofn.lpstrFile = szFileName;
	ofn.nMaxFile = MAX_PATH;
	ofn.Flags = OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
	ofn.lpstrDefExt = "bod";

	if(GetSaveFileName(&ofn))
	{
		f = fopen(szFileName, "w");
		if(f)
		{
			put_headers(hwnd, f);
			put_header(hwnd, f);
			for(i = 1; i < SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCOUNT, 0, 0) - 2; i++)
			{
				song *s = (song *) SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i, 0);
				put_tracklist_entry(hwnd, f, s);
			}

			for(i = 1; i < SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETCOUNT, 0, 0) - 2; i++)
			{
				song *s = (song *) SendDlgItemMessage(hwnd, IDC_TRACKS, LB_GETITEMDATA, i, 0);
				put_song(hwnd, f, s);
			}

			put_footer(hwnd, f);
			put_footers(hwnd, f);

			fclose(f);
		}
		else
		{
			MessageBox(hwnd, "Couldn't open file for writing", szFileName, MB_ICONERROR);
		}
	}
}


/* Da dialog procedure */
BOOL CALLBACK DlgProc(HWND hwnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
	switch(Message)
	{
		case WM_INITDIALOG:
			init_dialog(hwnd);
		break;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDC_GENERATE:
				generate_output(hwnd);
				break;
			case IDOK:
				add_song(hwnd);
				break;
			case IDC_UP:
				move_up(hwnd);
				break;
			case IDC_DOWN:
				move_down(hwnd);
				break;
			case IDC_TRACKS:
				{
					if(((int)HIWORD(wParam)) == LBN_SELCHANGE)
					{
						sel_change(hwnd);
					}
				}
				break;
			case IDC_REMOVE:
				{
					remove_track(hwnd);
				}
				break;
			}
		break;

		case WM_CLOSE:
			EndDialog(hwnd, 0);
		break;
/*	    case WM_SIZE:
	    {
		    HWND hEdit;
			RECT rcClient;

	        GetClientRect(hwnd, &rcClient);
	
		    hEdit = GetDlgItem(hwnd, IDC_EDIT);
			SetWindowPos(hEdit, NULL, 0, 0, rcClient.right, rcClient.bottom, SWP_NOZORDER);
	    }
*/
		default:
			return FALSE;
	}
	return TRUE;
}


int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
	DialogBox(hInstance, MAKEINTRESOURCE(IDD_MAIN), NULL, DlgProc);
	return 0;
}
